/**
 * Rezume Popup Script
 * Handles all popup UI logic:
 *   - Resume upload + login flow
 *   - JD extraction from current tab
 *   - Sending resume + JD to /api/match
 *   - Rendering match results
 */

'use strict';

// ---------------------------------------------------------------------------
// Config — update API_BASE for production deployment
// ---------------------------------------------------------------------------
const API_BASE = 'http://localhost:3000';

// ---------------------------------------------------------------------------
// SWE tech keywords — mirrored from content.js for client-side quick match.
// Used to compare what keywords appear in both the JD and the resume.
// ---------------------------------------------------------------------------
const SWE_TECH_KEYWORDS = [
  // Languages
  'python', 'javascript', 'typescript', 'java', 'c++', 'golang', 'go', 'rust',
  'ruby', 'php', 'swift', 'kotlin', 'scala', 'sql', 'html', 'css', 'bash', 'shell',
  'r language', 'matlab', 'haskell', 'elixir', 'clojure', 'groovy',
  // Frameworks & libraries
  'react', 'angular', 'vue', 'next.js', 'nuxt', 'svelte', 'node.js', 'express',
  'django', 'flask', 'fastapi', 'spring boot', 'spring', '.net', 'asp.net',
  'rails', 'laravel', 'symfony', 'tensorflow', 'pytorch', 'pandas', 'numpy',
  'scikit-learn', 'keras', 'jquery', 'tailwind',
  // Databases
  'postgresql', 'postgres', 'mysql', 'mongodb', 'redis', 'cassandra', 'dynamodb',
  'elasticsearch', 'oracle', 'sqlite', 'neo4j', 'couchdb', 'influxdb',
  'mariadb', 'sql server', 'snowflake', 'bigquery', 'redshift',
  // Cloud & infrastructure
  'aws', 'azure', 'gcp', 'google cloud', 'ec2', 's3', 'lambda', 'rds',
  'cloudformation', 'ecs', 'eks', 'cloudfront', 'vpc', 'iam',
  'heroku', 'digitalocean', 'vercel', 'netlify',
  // DevOps & tooling
  'docker', 'kubernetes', 'k8s', 'terraform', 'ansible', 'jenkins',
  'github actions', 'gitlab ci', 'circleci', 'travis ci', 'helm',
  'prometheus', 'grafana', 'datadog', 'new relic', 'splunk',
  'nginx', 'apache', 'git', 'github', 'gitlab', 'bitbucket',
  // Architecture & patterns
  'microservices', 'rest api', 'restful', 'graphql', 'grpc', 'kafka', 'rabbitmq',
  'ci/cd', 'serverless', 'api gateway', 'load balancing', 'message queue',
  'event-driven', 'service mesh', 'istio', 'caching', 'cdn',
  // Methodologies & practices
  'agile', 'scrum', 'kanban', 'devops', 'sre', 'tdd', 'bdd',
  'pair programming', 'continuous integration', 'continuous deployment', 'gitops',
  // Testing
  'jest', 'mocha', 'pytest', 'junit', 'selenium', 'cypress',
  'unit testing', 'integration testing', 'end-to-end testing',
  // Role titles that appear in JD bodies
  'software engineer', 'backend engineer', 'frontend engineer', 'full stack engineer',
  'devops engineer', 'data engineer', 'ml engineer', 'mobile engineer',
  'platform engineer', 'infrastructure engineer', 'site reliability engineer',
];

// Configure PDF.js worker to use the locally bundled file (MV3 requires this)
if (typeof pdfjsLib !== 'undefined') {
  pdfjsLib.GlobalWorkerOptions.workerSrc = chrome.runtime.getURL('vendor/pdf.worker.min.js');
}

// ---------------------------------------------------------------------------
// DOM refs
// ---------------------------------------------------------------------------
const viewLogin         = document.getElementById('viewLogin');
const viewMain          = document.getElementById('viewMain');
const userChip          = document.getElementById('userChip');
const userAvatar        = document.getElementById('userAvatar');
const userNameEl        = document.getElementById('userName');
const userEmailEl       = document.getElementById('userEmail');
const btnLogout         = document.getElementById('btnLogout');
const btnChangeResume   = document.getElementById('btnChangeResume');

// Login view
const dropZone          = document.getElementById('dropZone');
const resumeFileInput   = document.getElementById('resumeFileInput');
const loginFileName     = document.getElementById('loginFileName');
const btnLogin          = document.getElementById('btnLogin');

// Main view
const jdMeta            = document.getElementById('jdMeta');
const jdPreview         = document.getElementById('jdPreview');

// Quick match
const quickMatch        = document.getElementById('quickMatch');
const quickMatchFill    = document.getElementById('quickMatchFill');
const quickMatchPct     = document.getElementById('quickMatchPct');
const quickMatchDetail  = document.getElementById('quickMatchDetail');
const quickMatchPills   = document.getElementById('quickMatchPills');

// Results
const resultsSection    = document.getElementById('resultsSection');
const scoreRingFill     = document.getElementById('scoreRingFill');
const scoreValue        = document.getElementById('scoreValue');
const scoreLevel        = document.getElementById('scoreLevel');
const scoreMeta         = document.getElementById('scoreMeta');
const scoreVerdict      = document.getElementById('scoreVerdict');
const criticalGroup     = document.getElementById('criticalGroup');
const criticalPills     = document.getElementById('criticalPills');
const recommendedGroup  = document.getElementById('recommendedGroup');
const recommendedPills  = document.getElementById('recommendedPills');
const foundGroup        = document.getElementById('foundGroup');
const foundPills        = document.getElementById('foundPills');

// Status
const statusBar         = document.getElementById('statusBar');
const spinner           = document.getElementById('spinner');
const statusText        = document.getElementById('statusText');

// Tailor
const btnTailor         = document.getElementById('btnTailor');
const tailorSection     = document.getElementById('tailorSection');
const tailorProgress    = document.getElementById('tailorProgress');
const tailorProgressText = document.getElementById('tailorProgressText');
const tailorDone        = document.getElementById('tailorDone');
const tailorDoneLabel   = document.getElementById('tailorDoneLabel');
const btnDownloadTailor = document.getElementById('btnDownloadTailor');

// ---------------------------------------------------------------------------
// State
// ---------------------------------------------------------------------------
let selectedLoginFile   = null;   // File chosen for login
let extractedJD         = null;   // { text, url, title, method, confidence }
let cachedToken         = null;   // JWT stored in extension storage
let cachedUser          = null;   // { name, email }
let cachedResumeBase64  = null;   // Stored resume (base64)
let cachedResumeName    = null;   // Stored resume filename
let cachedResumeKeywords = null;  // Keywords extracted from the stored resume (array)
let tailoredPdfBlob     = null;   // Last tailored PDF blob (for download)

// ---------------------------------------------------------------------------
// Utility helpers
// ---------------------------------------------------------------------------

function bg(action, payload = {}) {
  return new Promise((resolve) =>
    chrome.runtime.sendMessage({ action, ...payload }, resolve)
  );
}

function show(el)  { el.classList.remove('hidden'); }
function hide(el)  { el.classList.add('hidden'); }

function setStatus(msg, type = 'loading') {
  show(statusBar);
  statusBar.className = `status-bar ${type}`;
  statusText.textContent = msg;
  if (type === 'loading') show(spinner);
  else hide(spinner);
}

function clearStatus() {
  hide(statusBar);
  hide(spinner);
}

function fileToBase64(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload  = () => resolve(reader.result.split(',')[1]); // strip data:... prefix
    reader.onerror = reject;
    reader.readAsDataURL(file);
  });
}

function base64ToBlob(b64, type = 'application/pdf') {
  const binary = atob(b64);
  const bytes  = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) bytes[i] = binary.charCodeAt(i);
  return new Blob([bytes], { type });
}

function initials(name) {
  if (!name) return '?';
  return name.split(' ').slice(0, 2).map(w => w[0]).join('').toUpperCase();
}

// ---------------------------------------------------------------------------
// Resume keyword extraction (client-side, using PDF.js)
// ---------------------------------------------------------------------------

/**
 * Extracts plain text from a base64-encoded PDF using the bundled PDF.js.
 */
async function extractTextFromPDF(base64) {
  const binary = atob(base64);
  const bytes  = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) bytes[i] = binary.charCodeAt(i);

  const loadingTask = pdfjsLib.getDocument({ data: bytes });
  const pdf         = await loadingTask.promise;

  let text = '';
  for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
    const page    = await pdf.getPage(pageNum);
    const content = await page.getTextContent();
    text += content.items.map(item => item.str).join(' ') + '\n';
  }
  return text;
}

/**
 * Returns the subset of SWE_TECH_KEYWORDS that appear in the given text.
 */
function extractKeywordsFromText(text) {
  const lower = text.toLowerCase();
  return SWE_TECH_KEYWORDS.filter(kw => lower.includes(kw));
}

/**
 * Kicks off async resume keyword extraction from the cached base64 PDF.
 * When done, calls tryShowQuickMatch in case the JD is already detected.
 */
async function extractResumeKeywords(base64) {
  try {
    const text = await extractTextFromPDF(base64);
    cachedResumeKeywords = extractKeywordsFromText(text);
    // If JD was already detected before keywords finished loading, show now
    if (extractedJD) tryShowQuickMatch(extractedJD.text);
  } catch (_) {
    cachedResumeKeywords = [];
  }
}

// ---------------------------------------------------------------------------
// Quick match calculation + rendering
// ---------------------------------------------------------------------------

/**
 * Computes overlap between JD keywords and resume keywords.
 * Returns null if either side has no keywords.
 */
function computeQuickMatch(jdText) {
  if (!cachedResumeKeywords || cachedResumeKeywords.length === 0) return null;

  const jdKeywords = extractKeywordsFromText(jdText);
  if (jdKeywords.length === 0) return null;

  const resumeSet = new Set(cachedResumeKeywords);
  const matched   = jdKeywords.filter(kw => resumeSet.has(kw));
  const score     = Math.round((matched.length / jdKeywords.length) * 100);

  return { score, matched, jdTotal: jdKeywords.length };
}

/**
 * Attempts a quick match and renders it if data is available.
 */
function tryShowQuickMatch(jdText) {
  const result = computeQuickMatch(jdText);
  if (result) renderQuickMatch(result);
}

/**
 * Renders the quick match bar, percentage, detail line and keyword pills.
 */
function renderQuickMatch({ score, matched, jdTotal }) {
  // Color based on score
  let color;
  if (score >= 70)      color = '#059669'; // green
  else if (score >= 40) color = '#2563eb'; // blue
  else                  color = '#d97706'; // amber

  quickMatchFill.style.width           = `${score}%`;
  quickMatchFill.style.backgroundColor = color;
  quickMatchPct.style.color            = color;
  quickMatchPct.textContent            = `${score}%`;

  const missing = jdTotal - matched.length;
  quickMatchDetail.textContent =
    `${matched.length} of ${jdTotal} JD keywords in your resume` +
    (missing > 0 ? ` · ${missing} missing` : ' · Full coverage!');

  // Keyword pills — show up to 5 matched, then "+N more"
  quickMatchPills.innerHTML = '';
  const shown = matched.slice(0, 5);
  shown.forEach(kw => {
    const span = document.createElement('span');
    span.className   = 'pill pill-qmatch';
    span.textContent = kw;
    quickMatchPills.appendChild(span);
  });
  if (matched.length > 5) {
    const more = document.createElement('span');
    more.className   = 'pill pill-qmatch-more';
    more.textContent = `+${matched.length - 5} more`;
    quickMatchPills.appendChild(more);
  }

  show(quickMatch);
}

// ---------------------------------------------------------------------------
// Score ring animation
// ---------------------------------------------------------------------------
function animateScore(score) {
  const circumference = 188.5; // 2 * PI * 30
  const offset = circumference - (score / 100) * circumference;

  // Color by score
  let color = '#dc2626'; // red
  if (score >= 75) color = '#059669';       // green
  else if (score >= 55) color = '#2563eb';  // blue
  else if (score >= 35) color = '#d97706';  // amber

  scoreRingFill.style.stroke = color;
  scoreRingFill.style.strokeDashoffset = offset;
  scoreValue.textContent = score;
}

function verdictFromScore(score) {
  if (score >= 75) return { label: 'Strong match',  cls: 'verdict-strong' };
  if (score >= 55) return { label: 'Good match',    cls: 'verdict-good'   };
  if (score >= 35) return { label: 'Fair match',    cls: 'verdict-fair'   };
  return             { label: 'Weak match',    cls: 'verdict-weak'   };
}

function renderPills(container, keywords, pillClass) {
  container.innerHTML = '';
  keywords.forEach(kw => {
    const span = document.createElement('span');
    span.className = `pill ${pillClass}`;
    span.textContent = kw;
    container.appendChild(span);
  });
}

// ---------------------------------------------------------------------------
// View switchers
// ---------------------------------------------------------------------------
function showLoginView() {
  hide(viewMain);
  hide(userChip);
  hide(btnLogout);
  show(viewLogin);
  clearStatus();
}

function showMainView(user) {
  hide(viewLogin);
  show(viewMain);
  show(userChip);
  show(btnLogout);

  // Populate user chip
  userAvatar.textContent = initials(user.name);
  userNameEl.textContent = user.name || 'Anonymous';
  userEmailEl.textContent = user.email || '';

  // Trigger JD extraction from current tab
  extractJDFromTab();
}

// ---------------------------------------------------------------------------
// Auth flow
// ---------------------------------------------------------------------------
async function loadAuthState() {
  const data = await bg('getAuth');
  cachedToken = data.token;
  cachedUser  = data.user;

  if (cachedToken && cachedUser) {
    // Load cached resume
    const resumeData = await bg('getResume');
    cachedResumeBase64 = resumeData.resumeBase64;
    cachedResumeName   = resumeData.resumeName;

    // Kick off background keyword extraction — non-blocking
    if (cachedResumeBase64) extractResumeKeywords(cachedResumeBase64);

    showMainView(cachedUser);
  } else {
    showLoginView();
  }
}

async function doLogin() {
  if (!selectedLoginFile) return;

  setStatus('Logging in...', 'loading');
  btnLogin.disabled = true;

  try {
    const formData = new FormData();
    formData.append('pdf', selectedLoginFile);

    const resp = await fetch(`${API_BASE}/api/auth/resume-login`, {
      method: 'POST',
      body: formData,
    });

    const json = await resp.json();

    if (!json.success) {
      setStatus(json.error || 'Login failed. Check the server is running.', 'error');
      btnLogin.disabled = false;
      return;
    }

    // Store token + user
    cachedToken = json.token;
    cachedUser  = json.user;
    await bg('saveAuth', { token: json.token, user: json.user });

    // Cache resume as base64 for future matching
    cachedResumeBase64 = await fileToBase64(selectedLoginFile);
    cachedResumeName   = selectedLoginFile.name;
    await bg('saveResume', { resumeBase64: cachedResumeBase64, resumeName: cachedResumeName });

    // Kick off background keyword extraction — non-blocking
    extractResumeKeywords(cachedResumeBase64);

    clearStatus();
    showMainView(cachedUser);

  } catch (err) {
    setStatus(`Network error: is the Rezume server running at ${API_BASE}?`, 'error');
    btnLogin.disabled = false;
  }
}

async function doLogout() {
  await bg('logout');
  cachedToken = cachedUser = cachedResumeBase64 = cachedResumeName = null;
  cachedResumeKeywords = null;
  selectedLoginFile = null;
  extractedJD = null;
  hide(resultsSection);
  hide(quickMatch);
  showLoginView();
}

// ---------------------------------------------------------------------------
// JD extraction
// ---------------------------------------------------------------------------

// URL schemes where content scripts cannot be injected
const RESTRICTED_PREFIXES = [
  'chrome://', 'chrome-extension://', 'edge://', 'about:', 'data:',
];

function isRestrictedUrl(url) {
  if (!url) return true;
  return RESTRICTED_PREFIXES.some(p => url.startsWith(p));
}

async function extractJDFromTab() {
  jdMeta.textContent = 'Extracting job description from page...';
  hide(jdPreview);
  hide(resultsSection);
  hide(quickMatch);
  clearStatus();

  console.log('[Rezume] extractJDFromTab() called');

  try {
    // In a side panel, currentWindow may not always resolve — fall back to lastFocusedWindow
    let [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    console.log('[Rezume] currentWindow query result:', tab ? `id=${tab.id} url=${tab.url}` : 'null');

    if (!tab) {
      [tab] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
      console.log('[Rezume] lastFocusedWindow fallback result:', tab ? `id=${tab.id} url=${tab.url}` : 'null');
    }

    if (!tab) {
      console.warn('[Rezume] No active tab found after both queries');
      jdMeta.textContent = 'No active tab found.';
      return;
    }

    // Browser-internal pages (chrome://, about:newtab, etc.) cannot run content scripts
    if (isRestrictedUrl(tab.url)) {
      console.log('[Rezume] Restricted URL — skipping injection:', tab.url);
      jdMeta.textContent = 'Open a job listing page to get started.';
      return;
    }

    console.log('[Rezume] Sending extractJD message to tab', tab.id, tab.url);

    // Try sending to an already-loaded content script first
    let response;
    try {
      response = await chrome.tabs.sendMessage(tab.id, { action: 'extractJD' });
      console.log('[Rezume] Got response from existing content script:', response?.success, 'method:', response?.method, 'textLen:', response?.text?.length);
    } catch (sendErr) {
      console.warn('[Rezume] sendMessage failed (no content script yet):', sendErr.message);
      // No listener — inject the content script manually, then retry
      try {
        console.log('[Rezume] Injecting content.js...');
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['content.js'],
        });
        console.log('[Rezume] Injection succeeded, retrying sendMessage...');
        response = await chrome.tabs.sendMessage(tab.id, { action: 'extractJD' });
        console.log('[Rezume] Got response after injection:', response?.success, 'method:', response?.method, 'textLen:', response?.text?.length);
      } catch (injectErr) {
        // Page still not injectable (e.g. PDF viewer, local file without permission)
        console.error('[Rezume] Injection also failed:', injectErr.message);
        jdMeta.textContent = 'Could not access this page. Try a job board like LinkedIn or Indeed.';
        return;
      }
    }

    if (response && response.success && response.text && response.text.length > 100) {
      console.log('[Rezume] JD extracted successfully. confidence:', response.confidence, 'chars:', response.text.length);
      extractedJD = response;

      const preview = response.text.substring(0, 300).replace(/\n+/g, ' ').trim();
      const confidenceLabel = response.confidence === 'high'   ? '✓'
                            : response.confidence === 'medium' ? '~'
                            : '';
      jdMeta.textContent = `${confidenceLabel} Detected on: ${response.title || tab.url}`.trim();
      jdPreview.textContent = preview + (response.text.length > 300 ? '...' : '');
      show(jdPreview);

      // Show quick match indicator if resume keywords are already ready
      tryShowQuickMatch(response.text);

      // Automatically run the full server-side match
      doMatch();
    } else {
      console.warn('[Rezume] Response did not contain usable JD text:', response);
      extractedJD = null;
      jdMeta.textContent = 'No job description found on this page.';
    }
  } catch (err) {
    console.error('[Rezume] Unexpected error in extractJDFromTab:', err.message, err.stack);
    extractedJD = null;
    jdMeta.textContent = 'Could not read this page. Try navigating to a job listing.';
  }
}

// ---------------------------------------------------------------------------
// Match resume to JD
// ---------------------------------------------------------------------------
async function doMatch() {
  if (!extractedJD) return;

  setStatus('Matching resume to job description...', 'loading');
  hide(resultsSection);

  try {
    if (!cachedResumeBase64) {
      setStatus('No resume available. Please login with your resume.', 'error');
      return;
    }

    const resumeBlob = base64ToBlob(cachedResumeBase64);
    const resumeName = cachedResumeName || 'resume.pdf';

    const formData = new FormData();
    formData.append('pdf', resumeBlob, resumeName);
    formData.append('jobDescription', extractedJD.text);

    const headers = {};
    if (cachedToken) headers['Authorization'] = `Bearer ${cachedToken}`;

    const resp = await fetch(`${API_BASE}/api/match`, {
      method: 'POST',
      headers,
      body: formData,
    });

    const json = await resp.json();

    if (!json.success) {
      setStatus(json.error || 'Match failed.', 'error');
      return;
    }

    clearStatus();
    renderResults(json.data);

  } catch (err) {
    setStatus(`Error: ${err.message}`, 'error');
  }
}

// ---------------------------------------------------------------------------
// Results rendering
// ---------------------------------------------------------------------------
function renderResults(data) {
  const analysis = data.analysis || {};
  const match    = data.match    || {};

  // Primary score: how well the resume matches THIS specific JD
  const matchScore = match.matchScore ?? 0;
  const atsScore   = analysis.atsScore ?? 0;
  animateScore(matchScore);

  // Level + meta line
  const levelLabel = (analysis.level || 'unknown').replace(/\b\w/g, c => c.toUpperCase());
  scoreLevel.textContent = levelLabel;

  const totalJDKw    = match.summary?.totalJDKeywords ?? 0;
  const totalMatched = match.summary?.totalMatched     ?? 0;
  scoreMeta.textContent = totalJDKw > 0
    ? `${totalMatched} of ${totalJDKw} JD keywords · ATS score: ${atsScore}`
    : `${analysis.keywordCount ?? 0} keywords found · ATS score: ${atsScore}`;

  const verdict = verdictFromScore(matchScore);
  scoreVerdict.className   = `score-verdict ${verdict.cls}`;
  scoreVerdict.textContent = verdict.label;

  // JD-specific missing keywords — what this job wants that the resume doesn't have
  const jdMissing = [...new Set(Object.values(match.missing?.byCategory || {}).flat())];
  if (jdMissing.length > 0) {
    renderPills(criticalPills, jdMissing.slice(0, 10), 'pill-critical');
    show(criticalGroup);
  } else {
    hide(criticalGroup);
  }

  // General ATS improvements — resume-level suggestions independent of this JD
  const atsImprovements = [
    ...(analysis.suggestions?.critical    || []),
    ...(analysis.suggestions?.recommended || []),
  ];
  if (atsImprovements.length > 0) {
    renderPills(recommendedPills, atsImprovements.slice(0, 8), 'pill-recommended');
    show(recommendedGroup);
  } else {
    hide(recommendedGroup);
  }

  // Keywords that ARE in both the resume and this JD
  const matchedInJD = [...new Set(Object.values(match.matched?.sample || {}).flat())];
  const foundKeywords = matchedInJD.length > 0
    ? matchedInJD
    : Object.values(analysis.foundKeywords || {}).flat().slice(0, 10);

  if (foundKeywords.length > 0) {
    renderPills(foundPills, [...new Set(foundKeywords)].slice(0, 10), 'pill-found');
    show(foundGroup);
  } else {
    hide(foundGroup);
  }

  show(resultsSection);
}

// ---------------------------------------------------------------------------
// PDF rendering helpers (ported from app.js, adapted for Blob input + 2× scale)
// ---------------------------------------------------------------------------

/**
 * Renders each page of a PDF blob to a PNG Blob using PDF.js.
 * Uses scale 2.0 (≈144 DPI) — good quality without excessive memory.
 * @param {Blob} pdfBlob
 * @param {number} [scale=2.0]
 * @returns {Promise<Blob[]>}
 */
async function renderPDFToImage(pdfBlob, scale = 2.0) {
  const arrayBuffer = await pdfBlob.arrayBuffer();
  const loadingTask = pdfjsLib.getDocument({ data: arrayBuffer });
  const pdf         = await loadingTask.promise;

  const imageBlobs = [];
  for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
    const page     = await pdf.getPage(pageNum);
    const viewport = page.getViewport({ scale });
    const canvas   = document.createElement('canvas');
    const ctx      = canvas.getContext('2d', { alpha: false });
    canvas.width   = viewport.width;
    canvas.height  = viewport.height;

    ctx.imageSmoothingEnabled  = true;
    ctx.imageSmoothingQuality  = 'high';

    await page.render({
      canvasContext: ctx,
      viewport,
      intent: 'print',
      renderInteractiveForms: false,
    }).promise;

    const blob = await new Promise((resolve, reject) => {
      canvas.toBlob(b => b ? resolve(b) : reject(new Error('Canvas to blob failed')), 'image/png', 1.0);
    });
    imageBlobs.push(blob);
  }
  return imageBlobs;
}

/**
 * Creates a PDF with a hidden text layer (ATS) and visible image pages.
 * @param {string}  text        - The tailored text (hidden layer)
 * @param {Blob[]}  imageBlobs  - Page images, one per page
 * @param {number}  [renderScale=2.0] - Scale used when rendering images (pixels-per-point)
 * @returns {Promise<Blob>}
 */
async function createEnhancedPDF(text, imageBlobs, renderScale = 2.0) {
  const { PDFDocument, rgb } = PDFLib;
  const pdfDoc = await PDFDocument.create();

  const pageWidth  = 612;  // US Letter: 8.5" × 11"
  const pageHeight = 792;
  const fontSize   = 10;
  const lineHeight = 12;
  const margin     = 50;

  const allLines           = text.split('\n');
  const totalPages         = imageBlobs.length;
  const linesPerPage       = Math.ceil(allLines.length / totalPages);

  for (let i = 0; i < totalPages; i++) {
    const page          = pdfDoc.addPage([pageWidth, pageHeight]);
    const imageAB       = await imageBlobs[i].arrayBuffer();
    const pngImage      = await pdfDoc.embedPng(imageAB);
    const imageDims     = pngImage.scale(1);

    // Convert pixel dimensions to PDF points using the render scale
    const imgWPts = imageDims.width  / renderScale;
    const imgHPts = imageDims.height / renderScale;
    const fitScale = Math.min(pageWidth / imgWPts, pageHeight / imgHPts);
    const finalW   = imgWPts * fitScale;
    const finalH   = imgHPts * fitScale;
    const x        = (pageWidth  - finalW) / 2;
    const y        = (pageHeight - finalH) / 2;

    // Hidden text layer
    const startLine = i * linesPerPage;
    const pageLines = allLines.slice(startLine, startLine + linesPerPage);
    let   curY      = pageHeight - margin;

    for (const line of pageLines) {
      if (line.trim()) {
        page.drawText(line.substring(0, 150), {
          x: margin, y: curY, size: fontSize, color: rgb(1, 1, 1),
        });
      }
      curY -= lineHeight;
      if (curY < margin) break;
    }

    // White cover + image overlay
    page.drawRectangle({ x: 0, y: 0, width: pageWidth, height: pageHeight, color: rgb(1, 1, 1) });
    page.drawImage(pngImage, { x, y, width: finalW, height: finalH });
  }

  const pdfBytes = await pdfDoc.save();
  return new Blob([pdfBytes], { type: 'application/pdf' });
}

// ---------------------------------------------------------------------------
// JD Tailoring
// ---------------------------------------------------------------------------

/**
 * Orchestrates the full tailor flow:
 *   1. POST resume + JD to /api/jd-tailor → tailoredText
 *   2. renderPDFToImage on the cached resume blob
 *   3. createEnhancedPDF(tailoredText, pageBlobs)
 *   4. Store blob + reveal download card
 */
async function doTailor() {
  if (!extractedJD) return;

  if (!cachedResumeBase64) {
    setStatus('No resume available. Please login with your resume.', 'error');
    return;
  }

  const resumeBlob = base64ToBlob(cachedResumeBase64);
  const resumeName = cachedResumeName || 'resume.pdf';

  // UI: show tailor section + progress
  show(tailorSection);
  show(tailorProgress);
  hide(tailorDone);
  tailorProgressText.textContent = 'Sending to AI for tailoring...';
  btnTailor.disabled = true;

  try {
    // Step 1: Get tailored text from server
    const formData = new FormData();
    formData.append('pdf', resumeBlob, resumeName);
    formData.append('jobDescription', extractedJD.text);

    const headers = {};
    if (cachedToken) headers['Authorization'] = `Bearer ${cachedToken}`;

    const resp = await fetch(`${API_BASE}/api/jd-tailor`, {
      method: 'POST',
      headers,
      body: formData,
    });

    const json = await resp.json();
    if (!json.success) {
      throw new Error(json.error || 'Tailoring failed');
    }

    const tailoredText = json.tailoredText;

    // Step 2: Render PDF pages to images
    tailorProgressText.textContent = 'Rendering resume pages...';
    const pageBlobs = await renderPDFToImage(resumeBlob, 2.0);

    // Step 3: Build the tailored PDF
    tailorProgressText.textContent = 'Building tailored PDF...';
    tailoredPdfBlob = await createEnhancedPDF(tailoredText, pageBlobs, 2.0);

    // Step 4: Show download card
    hide(tailorProgress);
    const baseName = resumeName.replace(/\.pdf$/i, '');
    tailorDoneLabel.textContent  = `tailored-${baseName}.pdf`;
    btnDownloadTailor.dataset.filename = `tailored-${baseName}.pdf`;
    show(tailorDone);

    tailorSection.scrollIntoView({ behavior: 'smooth', block: 'nearest' });

  } catch (err) {
    setStatus(`Tailoring failed: ${err.message}`, 'error');
    hide(tailorProgress);
  } finally {
    btnTailor.disabled = false;
  }
}

function downloadTailored() {
  if (!tailoredPdfBlob) return;
  const filename = btnDownloadTailor.dataset.filename || 'tailored-resume.pdf';
  const url = URL.createObjectURL(tailoredPdfBlob);
  const a   = document.createElement('a');
  a.href     = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  URL.revokeObjectURL(url);
  a.remove();
}

// ---------------------------------------------------------------------------
// Event listeners
// ---------------------------------------------------------------------------

// Drop zone click → trigger file input
dropZone.addEventListener('click', () => resumeFileInput.click());

// Drag and drop
dropZone.addEventListener('dragover', (e) => {
  e.preventDefault();
  dropZone.classList.add('drag-over');
});
dropZone.addEventListener('dragleave', () => dropZone.classList.remove('drag-over'));
dropZone.addEventListener('drop', (e) => {
  e.preventDefault();
  dropZone.classList.remove('drag-over');
  const file = e.dataTransfer.files[0];
  if (file && file.type === 'application/pdf') {
    onLoginFileSelected(file);
  }
});

resumeFileInput.addEventListener('change', () => {
  const file = resumeFileInput.files[0];
  if (file) onLoginFileSelected(file);
});

function onLoginFileSelected(file) {
  selectedLoginFile = file;
  loginFileName.textContent = file.name;
  show(loginFileName);
  btnLogin.disabled = false;
}

btnLogin.addEventListener('click', doLogin);
btnLogout.addEventListener('click', doLogout);

btnChangeResume.addEventListener('click', async () => {
  // Clear cached resume, return to login so user can upload a new one
  await bg('clearResume');
  cachedResumeBase64 = cachedResumeName = null;
  selectedLoginFile  = null;
  // Reset login form
  loginFileName.textContent = '';
  hide(loginFileName);
  btnLogin.disabled = true;
  showLoginView();
});

btnTailor.addEventListener('click', doTailor);
btnDownloadTailor.addEventListener('click', downloadTailored);

// ---------------------------------------------------------------------------
// Side panel: re-extract JD when the user switches tabs or navigates
// The panel stays alive across tab changes, so we need to watch for them.
// ---------------------------------------------------------------------------
chrome.tabs.onActivated.addListener(() => {
  if (cachedToken && cachedUser) extractJDFromTab();
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status !== 'complete') return;
  chrome.tabs.query({ active: true, currentWindow: true }, ([activeTab]) => {
    if (activeTab && activeTab.id === tabId && cachedToken && cachedUser) {
      extractJDFromTab();
    }
  });
});

// ---------------------------------------------------------------------------
// Init
// ---------------------------------------------------------------------------
loadAuthState();
