/**
 * Rezume Service Worker (background.js)
 * Handles token management and cross-tab coordination.
 * Runs as a Manifest V3 service worker.
 */

'use strict';

// ---------------------------------------------------------------------------
// Storage helpers — wraps chrome.storage.local in Promises
// ---------------------------------------------------------------------------
async function getStorage(keys) {
  return new Promise((resolve) => chrome.storage.local.get(keys, resolve));
}

async function setStorage(data) {
  return new Promise((resolve) => chrome.storage.local.set(data, resolve));
}

async function removeStorage(keys) {
  return new Promise((resolve) => chrome.storage.local.remove(keys, resolve));
}

// ---------------------------------------------------------------------------
// Message handler
// ---------------------------------------------------------------------------
chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  handleMessage(message).then(sendResponse).catch((err) => {
    sendResponse({ success: false, error: err.message });
  });
  return true; // Keep channel open for async response
});

async function handleMessage(message) {
  switch (message.action) {

    // ---- Save auth token + user after login --------------------------------
    case 'saveAuth': {
      const { token, user } = message;
      await setStorage({ rezume_token: token, rezume_user: user });
      return { success: true };
    }

    // ---- Load stored auth info --------------------------------------------
    case 'getAuth': {
      const data = await getStorage(['rezume_token', 'rezume_user']);
      return {
        success: true,
        token: data.rezume_token || null,
        user: data.rezume_user || null,
      };
    }

    // ---- Clear auth (logout) ---------------------------------------------
    case 'logout': {
      await removeStorage(['rezume_token', 'rezume_user', 'rezume_cached_resume']);
      return { success: true };
    }

    // ---- Save the user's resume PDF as base64 for reuse ------------------
    case 'saveResume': {
      const { resumeBase64, resumeName } = message;
      await setStorage({ rezume_cached_resume: resumeBase64, rezume_resume_name: resumeName });
      return { success: true };
    }

    // ---- Load cached resume ----------------------------------------------
    case 'getResume': {
      const data = await getStorage(['rezume_cached_resume', 'rezume_resume_name']);
      return {
        success: true,
        resumeBase64: data.rezume_cached_resume || null,
        resumeName: data.rezume_resume_name || null,
      };
    }

    // ---- Clear cached resume (user wants to upload a different one) -------
    case 'clearResume': {
      await removeStorage(['rezume_cached_resume', 'rezume_resume_name']);
      return { success: true };
    }

    default:
      return { success: false, error: `Unknown action: ${message.action}` };
  }
}

// ---------------------------------------------------------------------------
// Side panel — open when the action icon is clicked
// ---------------------------------------------------------------------------
chrome.action.onClicked.addListener((tab) => {
  chrome.sidePanel.open({ windowId: tab.windowId });
});

// ---------------------------------------------------------------------------
// On extension install / update
// ---------------------------------------------------------------------------
chrome.runtime.onInstalled.addListener(({ reason }) => {
  // Make clicking the toolbar icon open the side panel automatically
  chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true }).catch(() => {});

  if (reason === 'install') {
    console.log('[Rezume] Extension installed. Upload your resume to get started.');
  }
});
